<?php

namespace Pulse\Cti\Controller\AdminHtml\Form;

use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Api\Data\CustomerInterface;
use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrder;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\Data\OrderSearchResultInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactoryInterface;
use Psr\Log\LoggerInterface;

class Phone extends Action
{

    protected $resultJsonFactory;
    protected $customerRepository;
    protected $filterBuilder;
    protected $searchCriteriaBuilder;
    protected $orderRepository;
    protected $orderCollectionFactory;
    protected $logger;
    protected $sortOrderBuilder;

    public function __construct(
        Context $context,
        CustomerRepositoryInterface $customerRepository,
        JsonFactory $jsonFactory,
        FilterBuilder $filterBuilder,
        SortOrderBuilder $sortOrderBuilder,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        OrderRepositoryInterface $orderRepository,
        CollectionFactoryInterface $orderCollectionFactory,
        LoggerInterface $logger
    ) {
        parent::__construct($context);
        $this->customerRepository      = $customerRepository;
        $this->resultJsonFactory       = $jsonFactory;
        $this->filterBuilder           = $filterBuilder;
        $this->sortOrderBuilder        = $sortOrderBuilder;
        $this->searchCriteriaBuilder   = $searchCriteriaBuilder;
        $this->orderRepository         = $orderRepository;
        $this->orderCollectionFactory  = $orderCollectionFactory;
        $this->logger                  = $logger;
    }//end __construct()

    public function execute()
    {
        $phoneNumber = (array) $this->getRequest()->getPost('phone');

        $charFilter = [' ', '(', ')', '-'];

        $phoneNumber      = str_replace($charFilter, '', $phoneNumber[0]);
        $phoneNumberShort = $phoneNumber;

        if (substr($phoneNumber, 0, 1) === '+') {
            $phoneNumberShort = substr($phoneNumber, 3);
        }

        $resultJson = $this->resultJsonFactory->create();

        try {
            $customer = $this->getCustomerByPhone($phoneNumberShort);
            $order = $this->getLastOrder($customer->getId());
            $fullName = $this->getFullName($customer);
            $response = [
                'success' => 'true',
                'name'    => $fullName,
                'id'      => $customer->getId(),
                'order'   => (null === $order) ? 0 : $order->getId()
            ];
        } catch (\Exception $e) {
            $response = [
                'error'   => 'true',
                'message' => $e->getMessage(),
            ];
        }

        $this->_actionFlag->set('', self::FLAG_NO_POST_DISPATCH, true);
        return $resultJson->setData($response);
    }

    public function getFullName($customer): string
    {
        return implode(
            ' ',
            array_filter(
                [
                    $customer->getFirstname(),
                    $customer->getMiddlename(),
                    $customer->getLastName()
                ],
                function ($item) {
                    return ! empty($item);
                }
            )
        );
    }

    /**
     * Get CustomerInterface matching given phoneNumber
     *
     * @param string $phoneNumber
     *
     * @throws LocalizedException
     *
     * @return mixed
     */
    public function getCustomerByPhone(string $phoneNumber): CustomerInterface
    {
        $filter = $this->filterBuilder
            ->setField('billing_telephone')
            ->setConditionType('like')
            ->setValue('%' . $phoneNumber . '%')
            ->create();

        $this->searchCriteriaBuilder->addFilters([$filter]);
        $searchCriteria = $this->searchCriteriaBuilder->create();

        $customers = $this->customerRepository->getList($searchCriteria)->getItems();

        return $customers[0];
    }

    /**
     * @param int $customerId
     * @return OrderSearchResultInterface
     */
    public function getCustomerOrders(int $customerId): OrderSearchResultInterface
    {
        $this->searchCriteriaBuilder->addFilter(OrderInterface::CUSTOMER_ID, $customerId);
        $this->searchCriteriaBuilder->addFilter(OrderInterface::STATUS, ['canceled', 'holded'], 'nin');
        $searchCriteria = $this->searchCriteriaBuilder->create();
        $searchCriteria->setSortOrders(
            [
                $this->sortOrderBuilder
                    ->setField(OrderInterface::CREATED_AT)
                    ->setDirection(SortOrder::SORT_DESC)
                    ->create()
            ]
        );
        return $this->orderRepository->getList($searchCriteria);
    }

    /**
     * @param int $customerId
     * @return Order|null
     */
    public function getLastOrder(int $customerId): ?Order
    {
        $result = null;
        $searchResults = $this->getCustomerOrders($customerId);

        if ($searchResults->getTotalCount() > 0) {
            $orders = $searchResults->getItems();
            /** @var Order $lastOrder */
            $lastOrder = reset($orders);
            $result = $lastOrder;
        }
        return $result;
    }
}
